package gov.va.genisis2.DatabaseTableCopy;

import java.io.File;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.commons.dbutils.DbUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;

/**
 * This program copies a table from a source database to a temporary database to
 * have the table's details logged into a management table. From the temporary
 * database it copies the table to the destination before validating the table
 * on the destination side through a checksum and a row-count.
 * 
 * @author PII
 * 
 * 
 *         This program takes in the SQL server details in the order noted
 *         below.
 * 
 *         SourceIP SourceDB SourceSchema DestinationIP DestinationUsername
 *         DestinationPassword DestinationDB DestinationSchema ManagementIP
 *         ManagementUsername ManagementPassword ManagementDatabase
 *         TemporaryDatabase ManagementSchema TableName
 * 
 *         This program prints out a message either stating the table was not
 *         found, there was an error, or that it was successful.
 */
public class DatabaseTableCopy {
	private static final org.apache.logging.log4j.Logger logger = LogManager.getLogger(DatabaseTableCopy.class);

	public static void main(String[] args) {
		if (System.getProperty("config_home") != null) {
			File file = new File(System.getProperty("config_home"));
			if (file.isFile()) {
				LoggerContext context = (org.apache.logging.log4j.core.LoggerContext) LogManager.getContext(false);
				// this will force a reconfiguration
				context.setConfigLocation(file.toURI());
			} else {
				logger.error("Alternative logger properties not found. Using default properties.");
			}
		}
		logger.info("Arguments:");
		for (int i = 0; i < args.length; i++) {
			logger.info(i + ". " + args[i]);
		}
		// check for correct amount of arguments
		if (args.length != 15) {
			try {
				throw new IllegalArgumentException("Wrong amount of arguments.", null);
			} catch (IllegalArgumentException e1) {
				System.out.println("Wrong amount of arguments.");
				logger.error("Wrong amount of arguments.", e1);
				return;
			}
		}
		Boolean SourceSchemaCheck = false;
		Boolean SourceTableCheck = false;
		Boolean DestSchemaCheck = false;
		Boolean TableSchemaCheck = false;
		// initialize message to success
		StringBuilder message = new StringBuilder("Successfully copied table: ");
		// fill variables with arguments

		// source linked server on management
		String sourceIP = args[0].replaceAll("\'", "").trim();
		String sourceDB = args[1].replaceAll("\'", "").trim();
		String sourceSchema = args[2].replaceAll("\'", "").trim();
		// destination linked server on management and information to be able to
		// copy to destination
		String destIP = args[3].replaceAll("\'", "").trim();
		String destUsername = args[4].replaceAll("\'", "").trim();
		String destPassword = args[5].replaceAll("\'", "").trim();
		String destDB = args[6].replaceAll("\'", "").trim();
		String destSchema = args[7].replaceAll("\'", "").trim();
		// management linked server on destination and information to be able to
		// copy to management
		String mgmtIP = args[8].replaceAll("\'", "").trim();
		String mgmtUsername = args[9].replaceAll("\'", "").trim();
		String mgmtPassword = args[10].replaceAll("\'", "").trim();
		String mgmtDB = args[11].replaceAll("\'", "").trim();
		String tempDB = args[12].replaceAll("\'", "").trim();
		String mgmtSchema = args[13].replaceAll("\'", "").trim();
		String mgmtTable = "Management_table";
		// table name to be copied
		String tableName = args[14].replaceAll("\'", "").trim();
		if (logger.isInfoEnabled()) {
			logger.info("sourceIP " + sourceIP);
			logger.info("sourceDB " + sourceDB);
			logger.info("sourceSchema " + sourceSchema);
			logger.info("destIP " + destIP);
			logger.info("destUsername " + destUsername);
			logger.info("destPassword " + destPassword);
			logger.info("destDB " + destDB);
			logger.info("destSchema " + destSchema);
			logger.info("mgmtIP " + mgmtIP);
			logger.info("mgmtUsername " + mgmtUsername);
			logger.info("mgmtPassword " + mgmtPassword);
			logger.info("mgmtDB " + mgmtDB);
			logger.info("tempDB " + tempDB);
			logger.info("mgmtSchema " + mgmtSchema);
			logger.info("tableName " + tableName);
		}
		// initialize checksums
		BigInteger preChecksum;
		BigInteger postChecksum;
		Connection destCon = null;
		Connection mgmtCon = null;
		PreparedStatement GetSourceTables = null;
		PreparedStatement GetSourceSchemas = null;
		PreparedStatement GetSourceRowCount = null;
		PreparedStatement GetDestRowCount = null;
		PreparedStatement GetMaxID = null;
		PreparedStatement InsertManagementRow = null;
		PreparedStatement UpdateSourceRowCount = null;
		PreparedStatement UpdateDestRowCount = null;
		PreparedStatement GetSourceChecksum = null;
		PreparedStatement GetDestChecksum = null;
		PreparedStatement UpdateSourceChecksum = null;
		PreparedStatement UpdateDestChecksum = null;
		PreparedStatement mgmtCopy = null;
		PreparedStatement UpdateStatusBeforeCopy = null;
		PreparedStatement destCopy = null;
		PreparedStatement tempClean = null;
		PreparedStatement UpdateStatusAfterCopy = null;
		PreparedStatement UpdateCopyStatus = null;
		PreparedStatement CorruptedTable = null;
		ResultSet rs = null;
		try {
			logger.info("table copy process started");
			StringBuilder gst = new StringBuilder("EXEC sp_tables_ex @table_server = '" + sourceIP
					+ "',@table_catalog = '" + sourceDB + "',@table_type = 'TABLE'");
			StringBuilder gss = new StringBuilder("select * from [" + sourceIP + "].[" + sourceDB + "].sys.schemas");
			StringBuilder gsrc = new StringBuilder("select count(*) from [" + sourceIP + "].[" + sourceDB + "].["
					+ sourceSchema + "].[" + tableName + "]");
			StringBuilder gdrc = new StringBuilder(
					"select count(*) from [" + destDB + "].[" + destSchema + "].[" + tableName + "]");
			StringBuilder gmid = new StringBuilder(
					"SELECT MAX(ReqID) from [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable + "]");
			StringBuilder imr = new StringBuilder("INSERT INTO [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] (ReqID, DatabaseName, Source_Schema, Destination_Schema, Table_Name, dateandtime) VALUES ( ?,  '"
					+ sourceDB + "' , '" + sourceSchema + "' , '" + destSchema + "' , '" + tableName
					+ "', CURRENT_TIMESTAMP )");
			StringBuilder usrc = new StringBuilder("UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET rowcount_before_copy =? WHERE ReqID =?");
			StringBuilder udrc = new StringBuilder("UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET rowcount_after_copy =? WHERE ReqID =?");
			StringBuilder gscs = new StringBuilder("select SUM( CAST( checksum(*) AS BIGINT )) from [" + sourceIP
					+ "].[" + sourceDB + "].[" + sourceSchema + "].[" + tableName + "]");
			StringBuilder gdcs = new StringBuilder("select SUM( CAST( checksum(*) AS BIGINT )) from [" + destDB + "].["
					+ destSchema + "].[" + tableName + "]");
			StringBuilder uscs = new StringBuilder("UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET checksum_before_copy = ? WHERE ReqID = ?");
			StringBuilder udcs = new StringBuilder("UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET checksum_after_copy = ? WHERE ReqID = ?");
			StringBuilder mc = new StringBuilder("IF OBJECT_ID(N'[" + tempDB + "].[" + mgmtSchema + "].[" + tableName
					+ "]', N'U') IS NOT NULL DROP TABLE [" + tempDB + "].[" + mgmtSchema + "].[" + tableName
					+ "] SELECT * INTO [" + tempDB + "].[" + mgmtSchema + "].[" + tableName + "] FROM [" + sourceIP
					+ "].[" + sourceDB + "].[" + sourceSchema + "].[" + tableName + "]");
			StringBuilder usbc = new StringBuilder("update [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] set status_before_copy = 'Completed' where ReqID = ?");
			StringBuilder dc = new StringBuilder("IF OBJECT_ID(N'[" + destDB + "].[" + destSchema + "].[" + tableName
					+ "]', N'U') IS NOT NULL DROP TABLE [" + destDB + "].[" + destSchema + "].[" + tableName
					+ "] SELECT * INTO [" + destDB + "].[" + destSchema + "].[" + tableName + "] FROM [" + mgmtIP
					+ "].[" + tempDB + "].[" + mgmtSchema + "].[" + tableName + "]");
			StringBuilder tc = new StringBuilder("IF OBJECT_ID(N'[" + tempDB + "].[" + mgmtSchema + "].[" + tableName
					+ "]', N'U') IS NOT NULL DROP TABLE [" + tempDB + "].[" + mgmtSchema + "].[" + tableName + "]");
			StringBuilder usac = new StringBuilder("UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET status_after_copy = 'Completed' where ReqID = ?");
			StringBuilder ucs = new StringBuilder(" UPDATE [" + mgmtDB + "].[" + mgmtSchema + "].[" + mgmtTable
					+ "] SET CopyStatus ='Copy Successful' WHERE ReqID = ?");
			StringBuilder ct = new StringBuilder(" UPDATE [" + mgmtIP + "].[" + mgmtDB + "].[" + mgmtSchema + "].["
					+ mgmtTable + "] SET CopyStatus ='Copy Corrupted' WHERE ReqID = ? DROP TABLE IF EXISTS [" + destDB
					+ "].[" + destSchema + "].[" + tableName + "]");
			// Connect to server ... working on encryption
			destCon = DriverManager.getConnection("jdbc:jtds:sqlserver://" + destIP + "/" + destDB, destUsername,
					destPassword);
			mgmtCon = DriverManager.getConnection("jdbc:jtds:sqlserver://" + mgmtIP + "/" + mgmtDB, mgmtUsername,
					mgmtPassword);
			// initialize prepared statements for the current table
			GetSourceTables = mgmtCon.prepareStatement(gst.toString());
			GetSourceSchemas = mgmtCon.prepareStatement(gss.toString());
			GetSourceRowCount = mgmtCon.prepareStatement(gsrc.toString());
			GetDestRowCount = destCon.prepareStatement(gdrc.toString());
			GetMaxID = mgmtCon.prepareStatement(gmid.toString());
			InsertManagementRow = mgmtCon.prepareStatement(imr.toString());
			UpdateSourceRowCount = mgmtCon.prepareStatement(usrc.toString());
			UpdateDestRowCount = mgmtCon.prepareStatement(udrc.toString());
			GetSourceChecksum = mgmtCon.prepareStatement(gscs.toString());
			GetDestChecksum = destCon.prepareStatement(gdcs.toString());
			UpdateSourceChecksum = mgmtCon.prepareStatement(uscs.toString());
			UpdateDestChecksum = mgmtCon.prepareStatement(udcs.toString());
			mgmtCopy = mgmtCon.prepareStatement(mc.toString());
			UpdateStatusBeforeCopy = mgmtCon.prepareStatement(usbc.toString());
			destCopy = destCon.prepareStatement(dc.toString());
			tempClean = mgmtCon.prepareStatement(tc.toString());
			UpdateStatusAfterCopy = mgmtCon.prepareStatement(usac.toString());
			UpdateCopyStatus = mgmtCon.prepareStatement(ucs.toString());
			CorruptedTable = destCon.prepareStatement(ct.toString());

			// check for schema in source database
			rs = GetSourceSchemas.executeQuery();
			while (rs.next()) {
				if (sourceSchema.equals(rs.getString(1))) {
					SourceSchemaCheck = true;
				}
			}
			if (!SourceSchemaCheck) {
				System.out.println(sourceSchema
						+ " cannot be found. Please enter in a valid Source schema name for the table to be copied.");
				logger.error(sourceSchema
						+ " cannot be found. Please enter in a valid Source schema name for the table to be copied.");
				return;
			}
			DbUtils.close(rs);

			// check for table in source database
			rs = GetSourceTables.executeQuery();
			while (rs.next()) {
				if (tableName.equalsIgnoreCase(rs.getString("table_name"))) {
					SourceTableCheck = true;
				}
				if (tableName.equalsIgnoreCase(rs.getString("table_name"))
						&& sourceSchema.equals(rs.getString("table_schem"))) {
					TableSchemaCheck = true;
				}
			}
			if (!SourceTableCheck) {
				System.out.println(
						tableName + " cannot be found. Please enter in a valid table name for the table to be copied.");
				logger.error(
						tableName + " cannot be found. Please enter in a valid table name for the table to be copied.");
				return;
			} else if (!TableSchemaCheck) {
				System.out.println(tableName + " does not exist for the Schema " + sourceSchema + ".");
				logger.error(tableName + " does not exist for the Schema " + sourceSchema + ".");
				return;
			}
			DbUtils.close(rs);

			// check for schema in destination database
			rs = destCon.getMetaData().getSchemas();
			while (rs.next()) {
				if (destSchema.equals(rs.getString("TABLE_SCHEM"))) {
					DestSchemaCheck = true;
				}
			}
			if (!DestSchemaCheck) {
				System.out.println(destSchema
						+ " cannot be found. Please enter in a valid Destination schema name for the table to be copied.");
				logger.error(destSchema
						+ " cannot be found. Please enter in a valid Destination schema name for the table to be copied.");
				return;
			}
			DbUtils.close(rs);

			// get the current request ID from table by adding one to the
			// max
			rs = GetMaxID.executeQuery();
			rs.next();
			int maxID = rs.getInt(1) + 1;
			DbUtils.close(rs);
			// insert the row for the request into the management table
			InsertManagementRow.setInt(1, maxID);
			InsertManagementRow.executeUpdate();
			// count the number of rows before copying
			rs = GetSourceRowCount.executeQuery();
			rs.next();
			int preRowCount = rs.getInt(1);
			DbUtils.close(rs);
			UpdateSourceRowCount.setInt(1, preRowCount);
			UpdateSourceRowCount.setInt(2, maxID);
			UpdateSourceRowCount.executeUpdate();
			// calculate checksum before copying
			rs = GetSourceChecksum.executeQuery();
			rs.next();
			preChecksum = BigInteger.valueOf(rs.getLong(1));
			DbUtils.close(rs);
			UpdateSourceChecksum.setObject(1, preChecksum);
			UpdateSourceChecksum.setInt(2, maxID);
			UpdateSourceChecksum.executeUpdate();
			// copy the table into the database and update before copy
			mgmtCopy.executeUpdate();
			// status to success
			UpdateStatusBeforeCopy.setInt(1, maxID);
			UpdateStatusBeforeCopy.executeUpdate();

			destCopy.executeUpdate();
			tempClean.executeUpdate();
			// calculate checksum after copying
			rs = GetDestChecksum.executeQuery();
			rs.next();
			postChecksum = BigInteger.valueOf(rs.getLong(1));
			DbUtils.close(rs);
			UpdateDestChecksum.setObject(1, postChecksum);
			UpdateDestChecksum.setInt(2, maxID);
			UpdateDestChecksum.executeUpdate();
			// count the number of rows after copying
			rs = GetDestRowCount.executeQuery();
			rs.next();
			int postRowCount = rs.getInt(1);
			DbUtils.close(rs);
			UpdateStatusAfterCopy.setInt(1, maxID);
			UpdateStatusAfterCopy.executeUpdate();
			UpdateDestRowCount.setInt(1, postRowCount);
			UpdateDestRowCount.setInt(2, maxID);
			UpdateDestRowCount.executeUpdate();
			// update copy status
			UpdateCopyStatus.setInt(1, maxID);
			UpdateCopyStatus.executeUpdate();
			// check if the row numbers and checksums are equal if they
			// aren't, drop the corrupted copy and update copy status to
			// corrupted
			if (!(preChecksum.subtract(postChecksum)).equals(BigInteger.ZERO) || preRowCount - postRowCount != 0) {
				CorruptedTable.setInt(1, maxID);
				CorruptedTable.executeUpdate();
				throw new Exception("corruption on " + tableName, null);
			}
			// add the successful table to the message printed
			message.append("'" + tableName + "' ");
		}
		// generic catch for other errors
		catch (Exception e) {
			logger.error("An error occurred during transfer. Please contact your system administrator.", e);
			message = new StringBuilder("An error occurred during transfer. Please contact your system administrator.");
		} finally {
			try {
				DbUtils.close(rs);
				DbUtils.close(GetDestRowCount);
				DbUtils.close(GetMaxID);
				DbUtils.close(InsertManagementRow);
				DbUtils.close(UpdateSourceRowCount);
				DbUtils.close(UpdateDestRowCount);
				DbUtils.close(GetSourceChecksum);
				DbUtils.close(GetDestChecksum);
				DbUtils.close(UpdateSourceChecksum);
				DbUtils.close(UpdateDestChecksum);
				DbUtils.close(mgmtCopy);
				DbUtils.close(destCopy);
				DbUtils.close(tempClean);
				DbUtils.close(UpdateStatusAfterCopy);
				DbUtils.close(UpdateCopyStatus);
				DbUtils.close(CorruptedTable);
				DbUtils.close(destCon);
				DbUtils.close(mgmtCon);
			} catch (SQLException e) {
				logger.error("An error occurred during closing. Please contact your system administrator. ", e);
				message = new StringBuilder(
						"An error occurred during closing. Please contact your system administrator. ");
			}
		}
		logger.info("Output message: " + message.toString());
		System.out.println(message.toString());
	}

}
